#pragma	once

#ifdef MINISAX_EXPORTS
#define MINISAX_API __declspec(dllexport)
#else
#define MINISAX_API __declspec(dllimport)
#endif

namespace miniSAX {


	class MINISAX_API CIOException {
	public:
		CIOException(const char* aMessage);
		virtual ~CIOException();

		const char* getMessage() const { return message; }
	protected:
		char*	message;
	};


	class MINISAX_API CDataSource {
	public:
		CDataSource(const char* fileName) throw (CIOException);
		virtual ~CDataSource();

		virtual bool hasData() const;
		virtual char getChar();
		virtual void skipSpaces();

		virtual size_t	getDistance(char delimiter) const;
		virtual size_t	getNonElementDistance() const;

		virtual void getSequence(char* dest, int seqLen);

		virtual void back();

		virtual int getColumn() const { return col; }
		virtual int getRow() const { return row; }

	protected:
		char*	data;
		int		len;
		int		offset;

		int		col, row;

		virtual void freeData();
		virtual void loadFromFile(const char* fileName) throw (CIOException);
	};


	class MINISAX_API CMiniSAXException {
	public:
		/**
		 * Exception at the known position
		 */
		CMiniSAXException(const char* aMessage, int aCol, int aRow);
		/**
		 * Exception at an unknown position
		 */
		CMiniSAXException(const char* aMessage);
		virtual ~CMiniSAXException();

		const char* getMessage() const { return message; }
		int			getColumn()  const { return col;     }
		int			getRow()     const { return row;     }

	protected:
		char*	message;
		int		col;
		int		row;
	};


	class MINISAX_API CAttribute {
	public:
		CAttribute();
		CAttribute(const char* aName, const char* aValue);
		CAttribute(const CAttribute& instance);
		virtual ~CAttribute();

		virtual CAttribute& operator =(const CAttribute& instance);

		virtual const char* getName()  const { return name; }
		virtual const char* getValue() const { return value; }
	protected:
		char*	name;
		char*	value;

		virtual void freeMem();
		virtual void store(const char* aName, const char* aValue);
	};


	class MINISAX_API CAttributeList {
	public:
		CAttributeList();
		virtual ~CAttributeList();

		virtual void addAttribute(const CAttribute& newAttr);

		virtual size_t getCount() const { return count; }
		virtual const CAttribute* getAttribute(int index) const;
		virtual const CAttribute* getAttribute(const char* name) const;
	protected:
		CAttribute*	attributes;
		size_t		count;
	};


	class MINISAX_API CDocumentHandler {
	public:
		virtual void startDocument() throw (CMiniSAXException) = 0;
		virtual void endDocument() throw (CMiniSAXException) = 0;
	};


	class MINISAX_API CContentHandler {
	public:
		virtual void startElement(const char* name, const CAttributeList& attributes)
			 throw (CMiniSAXException) = 0;
		virtual void endElement(const char* name)
			 throw (CMiniSAXException) = 0;

		virtual void handleCharacters(const char* buffer, size_t len)
			 throw (CMiniSAXException) = 0;
	};


	class MINISAX_API CErrorHandler {
	public:
		virtual void handleError(const CMiniSAXException& e) = 0;
	};


	class MINISAX_API CEntityResolver {
	public:
		virtual void resolveEntity()
			throw (CMiniSAXException) = 0;
	};


	class MINISAX_API CMiniSAXParser {
	public:
		CMiniSAXParser();
		virtual ~CMiniSAXParser();

		virtual void setDocumentHandler(CDocumentHandler* aHandler) { documentHandler = aHandler; }
		virtual void setContentHandler(CContentHandler* aHandler) { contentHandler = aHandler; }
		virtual void setErrorHandler(CErrorHandler* aHandler) { errorHandler = aHandler; }
		virtual void setEntityResolver(CEntityResolver* aResolver) { entityResolver = aResolver; }

		virtual void parse(CDataSource* dataSource) throw(CMiniSAXException);

		virtual int getCurrentRow()    const { return currentRow;    }
		virtual int getCurrentColumn() const { return currentColumn; }

	protected:
		CDocumentHandler*	documentHandler;
		CContentHandler*	contentHandler;
		CErrorHandler*		errorHandler;
		CEntityResolver*	entityResolver;

		virtual bool readStartOfElement(CDataSource* dataSource);
		virtual bool readEndOfElement(CDataSource* dataSource);

		virtual bool readName(CDataSource* dataSource, char*& name);
		virtual bool readValue(CDataSource* dataSource, char*& value);

		virtual void startDocument() throw (CMiniSAXException);
		virtual void endDocument()   throw (CMiniSAXException);

		virtual void startElement(const char* name, const CAttributeList& attributes)
			throw (CMiniSAXException);
		virtual void endElement(const char* name)
			throw (CMiniSAXException);
		virtual void handleCharacters(const char* buffer, size_t len)
			throw (CMiniSAXException);

		virtual void handleError(const CMiniSAXException& e);

		/**
		 * These fields are accessible via corresponding get... methods.
		 * They are being updated each time before we transmit execution to the handler.
		 * This way handler can throw correct CMiniSAXException ;)
		 */
		int	currentRow, currentColumn;
	};

}